<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
if ( ! osf_is_woocommerce_activated() ) {
	return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class OSF_Elementor_Products_Categories extends Elementor\Widget_Base {

	public function get_categories() {
		return array( 'opal-addons' );
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve tabs widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'opal-product-categories';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve tabs widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Opal Product Categories', 'huntor-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve tabs widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}

	public static function get_button_sizes() {
		return [
			'xs' => __( 'Extra Small', 'huntor-core' ),
			'sm' => __( 'Small', 'huntor-core' ),
			'md' => __( 'Medium', 'huntor-core' ),
			'lg' => __( 'Large', 'huntor-core' ),
			'xl' => __( 'Extra Large', 'huntor-core' ),
		];
	}


	/**
	 * Register tabs widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		//Section Query
		$this->start_controls_section(
			'section_setting',
			[
				'label' => __( 'Settings', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'categories_name',
			[
				'label'       => __( 'Alternate Name', 'huntor-core' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => __( 'Alternate Name', 'huntor-core' ),
				'label_block' => true,
			]
		);
		$this->add_control(
			'icon',
			[
				'label'   => __( 'Choose Icon', 'huntor-core' ),
				'type'    => Controls_Manager::ICON
			]
		);

		$this->add_control(
			'categories',
			[
				'label'    => __( 'Categories', 'huntor-core' ),
				'type'     => Controls_Manager::SELECT2,
				'options'  => $this->get_product_categories(),
				'multiple' => false,
			]
		);

		$this->add_control(
			'categories_style',
			[
				'label'   => __( 'Style', 'huntor-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1' => __( 'Style 1', 'huntor-core' ),
					'2' => __( 'Style 2', 'huntor-core' ),
				],
			]
		);

		$this->add_control(
			'category_image',
			[
				'label'      => __( 'Choose Image', 'huntor-core' ),
				'default'    => [
					'url' => Elementor\Utils::get_placeholder_image_src(),
				],
				'type'       => Controls_Manager::MEDIA,
				'show_label' => false,
			]

		);

		$this->add_group_control(
			Elementor\Group_Control_Image_Size::get_type(),
			[
				'name'      => 'image',
				// Usage: `{name}_size` and `{name}_custom_dimension`, in this case `brand_image_size` and `brand_image_custom_dimension`.
				'default'   => 'full',
				'separator' => 'none',
			]
		);

		$this->add_control(
			'text_align_h',
			[
				'label'       => __( 'Horizontal Alignment', 'huntor-core' ),
				'type'        => Controls_Manager::CHOOSE,
				'label_block' => false,
				'default'     => 'center',
				'options'     => [
					'left'   => [
						'title' => __( 'Left', 'huntor-core' ),
						'icon'  => 'eicon-h-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'huntor-core' ),
						'icon'  => 'eicon-h-align-center',
					],
					'right'  => [
						'title' => __( 'Right', 'huntor-core' ),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'selectors'   => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'categories_style' => '1',
				]
			]
		);

		$this->add_control(
			'item_align_h',
			[
				'label'       => __( 'Horizontal Alignment', 'huntor-core' ),
				'type'        => Controls_Manager::CHOOSE,
				'label_block' => false,
				'default'     => 'center',
				'options'     => [
					'flex-start' => [
						'title' => __( 'Left', 'huntor-core' ),
						'icon'  => 'eicon-h-align-left',
					],
					'center'     => [
						'title' => __( 'Center', 'huntor-core' ),
						'icon'  => 'eicon-h-align-center',
					],
					'flex-end'   => [
						'title' => __( 'Right', 'huntor-core' ),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'selectors'   => [
					'{{WRAPPER}} .product-cats' => 'justify-content: {{VALUE}}',
				],
				'condition'   => [
					'categories_style' => '2',
				]
			]
		);

		$this->add_control(
			'text_align_v',
			[
				'label'        => __( 'Vertical Alignment', 'huntor-core' ),
				'type'         => Controls_Manager::CHOOSE,
				'label_block'  => false,
				'default'      => 'center',
				'options'      => [
					'flex-start' => [
						'title' => __( 'Top', 'huntor-core' ),
						'icon'  => 'eicon-v-align-top',
					],
					'center'     => [
						'title' => __( 'Middle', 'huntor-core' ),
						'icon'  => 'eicon-v-align-middle',
					],
					'flex-end'   => [
						'title' => __( 'Bottom', 'huntor-core' ),
						'icon'  => 'eicon-v-align-bottom',
					],
					'stretch'    => [
						'title' => __( 'Stretch', 'huntor-core' ),
						'icon'  => 'eicon-text-align-justify',
					],
				],
				'prefix_class' => 'elementor-vertical-align-',
				'selectors'    => [
					'{{WRAPPER}} .product-cats' => 'align-items: {{VALUE}}',
				],
				'condition'    => [
					'categories_style' => '2',
				]
			]
		);

		$this->add_control(
			'show_title',
			[
				'label' => __( 'Show Title', 'huntor-core' ),
				'type'  => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'show_number_item',
			[
				'label' => __( 'Show Number Item', 'huntor-core' ),
				'type'  => Controls_Manager::SWITCHER,
			]
		);


		$this->add_control(
			'show_button',
			[
				'label'     => __( 'Show Button', 'huntor-core' ),
				'type'      => Controls_Manager::SWITCHER,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'button_text',
			[
				'label'     => __( 'Text', 'huntor-core' ),
				'type'      => Controls_Manager::TEXT,
				'dynamic'   => [
					'active' => true,
				],
				'default'   => __( 'View Collection', 'huntor-core' ),
				'condition' => [
					'show_button' => 'yes',
				]
			]
		);

		$this->add_control(
			'button_type',
			[
				'label'        => __( 'Type', 'huntor-core' ),
				'type'         => Controls_Manager::SELECT,
				'default'      => 'primary',
				'options'      => [
					''                  => __( 'Default', 'huntor-core' ),
					'primary'           => __( 'Primary', 'huntor-core' ),
					'secondary'         => __( 'Secondary', 'huntor-core' ),
					'outline_primary'   => __( 'Outline Primary', 'huntor-core' ),
					'outline_secondary' => __( 'Outline Secondary', 'huntor-core' ),
					'dark'              => __( 'Dark', 'huntor-core' ),
					'light'             => __( 'Light', 'huntor-core' ),
					'link'              => __( 'Link', 'huntor-core' ),
				],
				'condition'    => [
					'show_button' => 'yes',
				],
				'prefix_class' => 'elementor-button-',
			]
		);

		$this->add_control(
			'button_size',
			[
				'label'          => __( 'Size', 'huntor-core' ),
				'type'           => Controls_Manager::SELECT,
				'default'        => 'md',
				'condition'      => [
					'show_button' => 'yes',
				],
				'options'        => self::get_button_sizes(),
				'style_transfer' => true,
			]
		);

		$this->end_controls_section();

		//STYLE
		$this->start_controls_section(
			'section_wrapper_style',
			[
				'label' => __( 'Wrapper', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
			'wrapper_pr_cate_padding',
			[
				'label'      => __( 'Padding', 'huntor-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .product-cats-meta' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
            'wrapper_pr_cate_margin',
            [
                'label'      => __( 'Margin', 'huntor-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .product-cats-meta' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
			'border_radius',
			[
				'label'      => __( 'Border Radius', 'huntor-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .product-cats-meta' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'wrapper_pr_cate_border',
				'selectors' => [
					'selector' => '{{WRAPPER}} .product-cats-meta',
				]

			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'image_box_shadow',
				'selector' => '{{WRAPPER}} .product-cats-meta',
			]
		);

        $this->add_control(
            'bg_color',
            [
                'label'     => __( 'Background', 'huntor-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .product-cats-meta' => 'background-color: {{VALUE}};',
                ],
            ]
        );



		$this->end_controls_section();

		$this->start_controls_section(
			'section_image_style',
			[
				'label' => __( 'Image', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'min-height',
			[
				'label'      => __( 'Min. Height', 'huntor-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 100,
						'max' => 1000,
					],
					'vh' => [
						'min' => 10,
						'max' => 100,
					],
				],
				'size_units' => [ 'px', 'vh' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-style-2' => 'min-height: {{SIZE}}{{UNIT}}',
				],
				'condition'  => [
					'categories_style' => '2'
				]
			]
		);

		$this->add_responsive_control(
			'img_spacing',
			[
				'label'      => __( 'Spacing', 'huntor-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .cats-image' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => __( 'Border Radius', 'huntor-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .cats-image img,{{WRAPPER}} .cats-image,{{WRAPPER}} .cats-image a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'effect_overlay',
			[
				'label'        => __( 'Effect', 'huntor-core' ),
				'type'         => Controls_Manager::SWITCHER,
				'return_value' => 'on',
				'prefix_class' => 'elementor-effect-',
			]
		);

        $this->start_controls_tabs( 'categories_colors' );

        $this->start_controls_tab(
            'categories_normal',
            [
                'label' => __( 'Normal', 'huntor-core' ),
            ]
        );

        $this->add_control(
            'overlay_color',
            [
                'label'     => __( 'Overlay Color', 'huntor-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cats-image:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_category_hover',
            [
                'label' => __( 'Hover', 'huntor-core' ),
            ]
        );


        $this->add_control(
            'overlay_color_hover',
            [
                'label'     => __( 'Overlay Color', 'huntor-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover .cats-image:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_name_style',
			[
				'label' => __( 'Name', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);



		$this->start_controls_tabs( 'tabs_name_style' );

		$this->start_controls_tab(
			'tab_name_normal',
			[
				'label' => __( 'Normal', 'huntor-core' ),
			]
		);

		$this->add_control(
			'name_color',
			[
				'label'     => __( 'Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product-cats:not(:hover) .cats-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_name_hover',
			[
				'label' => __( 'Hover', 'huntor-core' ),
			]
		);

		$this->add_control(
			'name_color_hover',
			[
				'label'     => __( 'Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product-cats:hover .cats-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'name_border',
				'selector' => '{{WRAPPER}} .product-cats:hover .cats-title a',

			]
		);

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'category_name_typography',
				'selector'  => '{{WRAPPER}} .cats-title',
				'label'     => 'Typography',
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'name_spacing',
			[
				'label'      => __( 'Spacing', 'huntor-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .cats-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_total_style',
			[
				'label' => __( 'Total', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
        $this->add_responsive_control(
            'icon_size',
            [
                'label'     => __( 'Icon Size', 'huntor-core' ),
                'type'      => Controls_Manager::SLIDER,
                'default'   => [
                    'size' => 14,
                ],
                'range'     => [
                    'px' => [
                        'min' => 6,
                        'max' => 300,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_space',
            [
                'label'     => __( 'Spacing', 'huntor-core' ),
                'type'      => Controls_Manager::SLIDER,
                'default'   => [
                    'size' => 5,
                ],
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .cats-total i' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label'     => __( 'Icon Color', 'huntor-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cats-total .elementor-icon' => 'color: {{VALUE}};',
                ],
            ]
        );


        $this->add_control(
			'total_color',
			[
				'label'     => __( 'Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cats-total' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'total_name_typography',
				'selector' => '{{WRAPPER}} .cats-total',
				'label'    => 'Typography'
			]
		);

		$this->add_responsive_control(
			'total_spacing',
			[
				'label'      => __( 'Spacing', 'huntor-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .cats-total' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'button_style',
			[
				'label'     => __( 'Button', 'huntor-core' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_button' => 'yes',
				]
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'typography',
				'selector' => '{{WRAPPER}} .elementor-button',
			]
		);

		$this->start_controls_tabs( 'tabs_button_style' );

		$this->start_controls_tab(
			'tab_button_normal',
			[
				'label' => __( 'Normal', 'huntor-core' ),
			]
		);

		$this->add_control(
			'button_text_color',
			[
				'label'     => __( 'Text Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .elementor-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_background_color',
			[
				'label'     => __( 'Background Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-button' => 'background-color: {{VALUE}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_button_hover',
			[
				'label' => __( 'Hover', 'huntor-core' ),
			]
		);

		$this->add_control(
			'hover_color',
			[
				'label'     => __( 'Text Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-button:hover' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'button_background_color_hover',
			[
				'label'     => __( 'Background Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_border_color_hover',
			[
				'label'     => __( 'Border Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->end_controls_tabs();


		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'button_border',
				'selector'  => '{{WRAPPER}} .elementor-button',
				'separator' => 'before',

			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'button_box_shadow',
				'selector' => '{{WRAPPER}} .elementor-button',
			]
		);
		$this->add_responsive_control(
			'button_text_padding',
			[
				'label'      => __( 'Padding', 'huntor-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'button_text_margin',
			[
				'label'      => __( 'Margin', 'huntor-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

	}


	protected function get_product_categories() {
		$categories = get_terms( array(
				'taxonomy'   => 'product_cat',
				'hide_empty' => false,
			)
		);
		$results    = array();
		if ( ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$results[ $category->slug ] = $category->name;
			}
		}

		return $results;
	}

	/**
	 * Render tabs widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_render_attribute( 'wrapper', 'class', 'product-cats' );
		$this->add_render_attribute( 'wrapper', 'class', 'elementor-style-' . esc_attr( $settings['categories_style'] ) );

		if ( empty( $settings['categories'] ) ) {
			return;
		}
		$has_icon = ! empty( $settings['icon'] );

		$this->add_render_attribute( 'icon', 'class', 'elementor-icon' );

		if ( $has_icon ) {
			$this->add_render_attribute( 'i', 'class', $settings['icon'] );
			$this->add_render_attribute( 'i', 'aria-hidden', 'true' );
		}

		$category = get_term_by( 'slug', $settings['categories'], 'product_cat' );
		if ($category && ! is_wp_error( $category ) ) {
			$image = '';
			if ( ! empty( $settings['category_image']['id'] ) ) {
				$image = Group_Control_Image_Size::get_attachment_image_src( $settings['category_image']['id'], 'image', $settings );
			} else {
				$thumbnail_id = get_woocommerce_term_meta( $category->term_id, 'thumbnail_id', true );
				if ( ! empty( $thumbnail_id ) ) {
					$image = wp_get_attachment_url( $thumbnail_id );
				}
			}
			?>

            <div <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
				<?php if ( $image ): ?>
                    <div class="cats-image">
                        <img src="<?php echo esc_url_raw( $image ); ?>"
                             alt="<?php echo esc_html( $category->name ); ?>">
                        <a href="<?php echo esc_url( get_term_link( $category ) ); ?>"></a>
                    </div>
				<?php endif; ?>
                <div class="product-cats-meta">
                    <a class="link_category_product" href="<?php echo esc_url( get_term_link( $category ) ); ?>"
                       title="<?php echo esc_attr( $category->name ); ?>">
                        <span class="screen-reader-text"><?php echo esc_html( $category->name ); ?></span>
                    </a>

                    <div class="cats-header">
						<?php if ( $settings['show_title'] ): ?>
                            <h2 class="cats-title">
                                <a href="<?php echo esc_url( get_term_link( $category ) ); ?>"
                                   title="<?php echo esc_attr( $category->name ); ?>">

                                    <span class="cats-title-text"><?php echo empty( $settings['categories_name'] ) ? esc_html( $category->name ) : wp_kses_post( $settings['categories_name'] ); ?></span>
                                </a>
                            </h2>
						<?php endif; ?>

						<?php if ( $settings['show_number_item'] ): ?>
                            <div class="cats-total">
                                <span ><?php echo esc_html( $category->count ) . esc_html__( ' products', 'huntor-core' ); ?></span>
                            <?php if ( $has_icon ): ?>
                                <span <?php echo $this->get_render_attribute_string( 'icon' ); ?>>
                                        <i <?php echo $this->get_render_attribute_string( 'i' ); ?>></i>
                                    </span>
                            <?php endif; ?>
                            </div>
						<?php endif; ?>
                    </div>

					<?php
					if ( $settings['show_button'] === 'yes' ) {
						?>
                        <div class="cats-button">
                            <a class="elementor-button elementor-size-<?php echo $settings['button_size']; ?>"
                               href="<?php echo esc_url( get_term_link( $category ) ); ?>"
                               title="<?php echo esc_attr( $category->name ); ?>">
								<?php echo esc_html( $settings['button_text'] ); ?>
                            </a>
                        </div>
						<?php
					}
					?>
                </div>
            </div>

			<?php

		}

	}
}

$widgets_manager->register( new OSF_Elementor_Products_Categories() );

