<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
if ( ! osf_is_woocommerce_activated() ) {
	return;
}

use Elementor\Controls_Manager;

/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class OSF_Elementor_Product_Deals extends OSF_Elementor_Carousel_Base {


	public function get_categories() {
		return array( 'opal-addons' );
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve tabs widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'opal-product-deals';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve tabs widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Opal Product Deals', 'huntor-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve tabs widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}


	public static function get_button_sizes() {
		return [
			'xs' => __( 'Extra Small', 'huntor-core' ),
			'sm' => __( 'Small', 'huntor-core' ),
			'md' => __( 'Medium', 'huntor-core' ),
			'lg' => __( 'Large', 'huntor-core' ),
			'xl' => __( 'Extra Large', 'huntor-core' ),
		];
	}

	/**
	 * Register tabs widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function _register_controls() {

		//Section Query
		$this->start_controls_section(
			'section_setting',
			[
				'label' => __( 'Settings', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);


		$this->add_control(
			'limit',
			[
				'label'   => __( 'Posts Per Page', 'huntor-core' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 6,
			]
		);

		$this->add_responsive_control(
			'column',
			[
				'label'   => __( 'columns', 'huntor-core' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 3,
				'options' => [ 1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6 ],
			]
		);


		$this->add_control(
			'advanced',
			[
				'label' => __( 'Advanced', 'huntor-core' ),
				'type'  => Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'   => __( 'Order By', 'huntor-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => [
					'date'       => __( 'Date', 'huntor-core' ),
					'id'         => __( 'Post ID', 'huntor-core' ),
					'menu_order' => __( 'Menu Order', 'huntor-core' ),
					'popularity' => __( 'Number of purchases', 'huntor-core' ),
					'rating'     => __( 'Average Products Rating', 'huntor-core' ),
					'title'      => __( 'Products Title', 'huntor-core' ),
					'rand'       => __( 'Random', 'huntor-core' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'   => __( 'Order', 'huntor-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc'  => __( 'ASC', 'huntor-core' ),
					'desc' => __( 'DESC', 'huntor-core' ),
				],
			]
		);

		$this->add_control(
			'categories',
			[
				'label'    => __( 'Categories', 'huntor-core' ),
				'type'     => Controls_Manager::SELECT2,
				'options'  => $this->get_product_categories(),
				'multiple' => true,
			]
		);

		$this->add_control(
			'cat_operator',
			[
				'label'     => __( 'Category Operator', 'huntor-core' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'IN',
				'options'   => [
					'AND'    => __( 'AND', 'huntor-core' ),
					'IN'     => __( 'IN', 'huntor-core' ),
					'NOT IN' => __( 'NOT IN', 'huntor-core' ),
				],
				'condition' => [
					'categories!' => ''
				],
			]
		);

		$this->add_responsive_control(
			'product_gutter',
			[
				'label'      => __( 'Gutter', 'huntor-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} ul.products li.product' => 'padding-left: calc({{SIZE}}{{UNIT}} / 2); padding-right: calc({{SIZE}}{{UNIT}} / 2);',
					'{{WRAPPER}} ul.products'            => 'margin-left: calc({{SIZE}}{{UNIT}} / -2); margin-right: calc({{SIZE}}{{UNIT}} / -2);',
				],
			]
		);

		$this->end_controls_section();
		// End Section Query

		$this->start_controls_section(
			'section_product_style',
			[
				'label' => __( 'Product', 'huntor-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'product_border_color',
			[
				'label'     => __( 'Product Items Border Color', 'huntor-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-product-list li .inner' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		// Carousel Option
		$this->add_control_carousel();
	}


	protected function get_product_categories() {
		$categories = get_terms( array(
				'taxonomy'   => 'product_cat',
				'hide_empty' => false,
			)
		);
		$results    = array();
		if ( ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$results[ $category->slug ] = $category->name;
			}
		}

		return $results;
	}


	/**
	 * Render tabs widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$this->woocommerce_default( $settings );
	}

	private function woocommerce_default( $settings ) {

		$atts = [
			'limit'          => $settings['limit'],
			'columns'        => $settings['column'],
			'orderby'        => $settings['orderby'],
			'order'          => $settings['order'],
			'product_layout' => 'list',
		];

		$type = 'sale_products';


		if ( ! empty( $settings['categories'] ) ) {
			$atts['category']     = implode( ',', $settings['categories'] );
			$atts['cat_operator'] = $settings['cat_operator'];
		}

		$atts['show_category']  = false;
		$atts['show_rating']    = true;
		$atts['show_time_sale'] = true;

		// Carousel
		if ( $settings['enable_carousel'] === 'yes' ) {
			$atts['carousel_settings'] = json_encode( wp_slash( $this->get_carousel_settings() ) );
			$atts['product_layout']    = 'list-carousel';
			$atts['class']             = 'woocommerce-product-list';
		} else {
			$atts['class'] = '';

			if ( ! empty( $settings['column_tablet'] ) ) {
				$atts['class'] .= ' columns-tablet-' . $settings['column_tablet'];
			}

			if ( ! empty( $settings['column_mobile'] ) ) {
				$atts['class'] .= ' columns-mobile-' . $settings['column_mobile'];
			}
		}

		$shortcode = new WC_Shortcode_Products( $atts, $type );

		echo $shortcode->get_content();
	}
}

$widgets_manager->register_widget_type( new OSF_Elementor_Product_Deals() );